#!/bin/bash
#
# /etc/init.d script for starting bc-agent as a service.
#
### BEGIN INIT INFO
#
# Provides: bc-agent
# Default-Start: 2 3 4 5
# Default-Stop: 0 1 6
# Description: BeyondCron agent
# Required-Start: $local_fs $named $network
# Required-Stop: $local_fs $named $network
#
### END INIT INFO
#
# This script will source the following files if they exist in order, allowing
# the user, host and server names to be specified as environment variables
# that are passed through to bc-agent.
#
#  1: /etc/bc-agent/service.profile
#  2: ~/.bc-agent
#
# Copyright 2023 BeyondCron Pty Ltd and related entities / All rights reserved.

export PATH=/bin:/usr/bin

main() {
  initVars

  case "$1" in
  start)
    start
    ;;
  stop)
    stop
    ;;
  restart|force-reload)
    stop
    start
    ;;
  status)
    status
    ;;
  *)
    echo "Usage: $0 (start|stop|restart|force-reload|status)" >&2
    exit 3
  esac

  exit $?
}

initVars() {
  class='com.beyondcron.agent.App'
  package='bc-agent'

  self=$(realPath $0)
  dir=$(dirname ${self})
  name=$(basename ${self})

  processPattern="java( .*)? -classpath .*/${name}/lib/${name}-"

  cmd=$(realPath "${dir}/../bin/${name}")
  if [ ! -x "${cmd}" ]; then
    echo "$0: ${cmd} does not exist" >&2
    exit 1
  fi

  loadConfig "/etc/${name}.conf"

  if [ -n "${BEYONDCRON_OS_USER}" ]; then
    user=${BEYONDCRON_OS_USER}
  else
    user=$(id -un)
  fi
  uid=$(id -u $user)
  eval home=~$user

  loadConfig "${home}/.${name}"

  if [ $uid -eq 0 ]; then
    echo "$0: run as a non-root user" >&2
    exit 1
  fi

  stopSignal=${BEYONDCRON_SIGNAL_STOP:-WINCH}
  if [ $(kill -l | grep -c ${stopSignal}) -ne 1 ]; then
    echo "$0: unknown stop signal ${stopSignal}, using WINCH instead" >&2
    stopSignal=WINCH
  fi
}

initLogging() {
  if [ -n "${BEYONDCRON_LOG_FILE}" ]; then
    touch ${BEYONDCRON_LOG_FILE} > /dev/null 2>&1
    if [ $? -eq 0 ]; then
      log=${BEYONDCRON_LOG_FILE};
    fi
  fi

  if [ -z "${log}" ]; then
    log=/var/log/${name}.log
    touch ${log} > /dev/null 2>&1
    if [ $? -ne 0 ]; then
      log=/tmp/${name}.${user}.log
    fi
  fi

  if [ -n "${BEYONDCRON_LOG_FILE}" -a "${BEYONDCRON_LOG_FILE}" != "${log}" ]; then
    echo "$0: cannot write to ${BEYONDCRON_LOG_FILE}, logging to ${log} instead" >&2
  fi

  if [ -s ${log} ]; then
    mv ${log} ${log}.old
  fi

  exec > ${log}
  exec 2>&1
}

loadConfig() {
  if [ -f "$1" ]; then
    echo "Loading config file $1"
    source $1
  fi
}

isRunning() {
  pgrep -f -U ${uid} "${processPattern}" > /dev/null 2>&1
  return $?
}

start() {
  isRunning
  if [ $? -ne 0 ]; then
    echo "Starting ${name}"
    initLogging
    if [ $uid -ne $(id -u) ]; then
      sudo -b -E -n -u $user ${cmd}
    else
      ${cmd}  &
    fi

    if [ $? -ne 0 ]; then
    echo "Could not start ${name}" >&2
      return 1
    fi
  else
    echo "${name} is already running" >&2
  fi

  return 0
}

stop() {
  isRunning
  if [ $? -eq 0 ]; then
    echo "Stopping ${name}"
    pkill -${stopSignal} -f -U ${uid} "${processPattern}"

    if [ $? -ne 0 ]; then
      echo "Could not stop ${name}" >&2
      return 1
    fi

    # wait up to 10 seconds for process to stop
    i=10
    while [ $i -gt 0 ]; do
      isRunning
      if [ $? -ne 0 ]; then
        return 0
      fi
      sleep 1
      ((--i))
    done

    return 1

  else
    echo "${name} is not running" >&2
    return 0
  fi
}

status() {
  isRunning
  if [ $? -ne 0 ]; then
    echo "${name} is not running"
    return 3
  else
    echo "${name} is running"
    return 0
  fi
}

realPath() {
  path=$1
  target=$(readlink ${path})

  while [ -n "${target}" ]; do
    path=$(cd $(dirname ${path}) && cd $(dirname ${target}) && pwd -P)/$(basename ${target})
    target=$(readlink ${path})
  done

  if [ -d {$path} ]; then
    path=$(cd ${path} && pwd -P)
  elif [ -f ${path} ]; then
    path=$(cd $(dirname ${path}) && pwd -P)/$(basename ${path})
  fi

  echo ${path}
}

main "$@"

# end