#!/bin/bash
#
# /etc/init.d script for starting bc-server as a service.
#
### BEGIN INIT INFO
#
# Provides: bd-server
# Default-Start: 2 3 4 5
# Default-Stop: 0 1 6
# Description: BeyondCron server
# Required-Start: $local_fs $named $network
# Required-Stop: $local_fs $named $network
#
### END INIT INFO
#
# This script will source the following files if they exist in order, allowing
# the server directory to be specified as an environment variable.
#
#  1: /etc/bc-server/service.profile
#  2: ~/.bc-server
#
# To start the server under a different user id define the BEYONDCRON_OS_USER variable within one of the above
# configuration files, or as an environment variable. e.g.
#
#   BEYONDCRON_OS_USER=beyondcron
#
# To force start the server, add the "--force" argument when starting the server. e.g.
#
#   /etc/init.d/bc-server start --force
#
# Copyright 2023 BeyondCron Pty Ltd and related entities / All rights reserved.

export PATH=/bin:/usr/bin

main() {
  initVars

  case "$1" in
  start)
    start $2
    ;;
  stop)
    stop
    ;;
  restart|force-reload)
    stop
    start
    ;;
  status)
    status
    ;;
  *)
    echo "Usage: $0 (start|stop|restart|force-reload|status)" >&2
    exit 1
  esac

  exit $?
}

initVars() {
  class='com.beyondcron.server.App'

  self=$(realPath $0)
  dir=$(dirname ${self})
  name=$(basename ${self})

  processPattern="java( .*)? -classpath .*/${name}/lib/${name}-"

  cmd=$(realPath "${dir}/../bin/${name}")
  if [ ! -x ${cmd} ]; then
    echo "$0: ${cmd} does not exist" >&2
    exit 1
  fi

  loadConfig "/etc/${name}.conf"

  if [ -n "${BEYONDCRON_OS_USER}" ]; then
    user=${BEYONDCRON_OS_USER}
  else
    user=$(id -un)
  fi
  uid=$(id -u $user)
  eval home=~$user

  loadConfig "${home}/.${name}"

  if [ $uid -eq 0 ]; then
    echo "$0: run as a non-root user" >&2
    exit 1
  fi

  if [ -z "${BEYONDCRON_DIR}" ]; then
    if [ -z "${BEYONDCRON_DIR_CONFIG}" -o -z "${BEYONDCRON_DIR_DATA}" ]; then
      echo "$0: environment variable BEYONDCRON_DIR or" >&2
      echo "    environment variables BEYONDCRON_DIR_CONFIG & BEYONDCRON_DIR_DATA undefined" >&2
      exit 1;
    fi
  fi

  stopSignal=${BEYONDCRON_SIGNAL_STOP:-WINCH}
  if [ $(kill -l | grep -c ${stopSignal}) -ne 1 ]; then
    echo "$0: unknown stop signal ${stopSignal}, using WINCH instead" >&2
    stopSignal=WINCH
  fi
}

initLogging() {
  if [ -n "${BEYONDCRON_LOG_FILE}" ]; then
    touch ${BEYONDCRON_LOG_FILE} > /dev/null 2>&1
    if [ $? -eq 0 ]; then
      log=${BEYONDCRON_LOG_FILE};
    fi
  fi

  if [ -z "${log}" ]; then
    log=/var/log/${name}.log
    touch ${log} > /dev/null 2>&1
    if [ $? -ne 0 ]; then
      log=/tmp/${name}.${user}.log
    fi
  fi

  if [ -n "${beyondCron_log_file}" -a "${beyondCron_log_file}" != "${log}" ]; then
    echo "$0: cannot write to ${beyondCron_log_file}, logging to ${log} instead" >&2
  fi

  if [ -s ${log} ]; then
    mv ${log} ${log}.old
  fi

  exec > ${log}
  exec 2>&1
}

loadConfig() {
  if [ -f "$1" ]; then
    echo "Loading config file $1"
    source $1
  fi
}

isRunning() {
  pgrep -f -U ${uid} "${processPattern}" > /dev/null 2>&1
  return $?
}

start() {
  isRunning
  if [ $? -ne 0 ]; then
    if [ "$1" = '--force' ]; then
      echo "Force starting ${name}"
      export BEYONDCRON_FORCE_START=true
    else
      echo "Starting ${name}"
    fi

    initLogging
    if [ $uid -ne $(id -u) ]; then
      if [ -z "${BEYONDCRON_FORCE_START}" ]; then
        sudo -b -E -n -u $user ${cmd}
      else
        sudo -b -E -n -u $user BEYONDCRON_FORCE_START=${BEYONDCRON_FORCE_START} ${cmd}
      fi
    else
      ${cmd} &
    fi

    if [ $? -ne 0 ]; then
    echo "Could not start ${name}" >&2
      return 1
    fi
  else
    echo "${name} is already running" >&2
  fi

  return 0
}

stop() {
  isRunning
  if [ $? -eq 0 ]; then
    echo "Stopping ${name}"
    pkill -${stopSignal} -f -U ${uid} "${processPattern}"

    if [ $? -ne 0 ]; then
      echo "Could not stop ${name}" >&2
      return 1
    fi

    # wait up to 10 seconds for process to stop
    i=10
    while [ $i -gt 0 ]; do
      isRunning
    if [ $? -ne 0 ]; then
      return 0
    fi
      sleep 1
      ((--i))
    done

    return 1

  else
    echo "${name} is not running" >&2
    return 0
  fi
}

status() {
  isRunning
  if [ $? -ne 0 ]; then
    echo "${name} is not running"
    return 3
  fi
  echo "${name} is running"
  return 0
}

realPath() {
  path=$1
  target=$(readlink ${path})

  while [ -n "${target}" ]; do
    path=$(cd $(dirname ${path}) && cd $(dirname ${target}) && pwd -P)/$(basename ${target})
    target=$(readlink ${path})
  done

  if [ -d {$path} ]; then
    path=$(cd ${path} && pwd -P)
  elif [ -f ${path} ]; then
    path=$(cd $(dirname ${path}) && pwd -P)/$(basename ${path})
  fi

  echo ${path}
}

main "$@"

# end