/*
 * Decompiled with CFR 0.152.
 */
package com.beyondcron.core;

import com.beyondcron.core.Localise;
import com.beyondcron.core.LogUtils;
import com.beyondcron.core.Program;
import com.beyondcron.core.StringUtils;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.UserPrincipal;
import java.util.regex.Pattern;
import org.apache.logging.log4j.Logger;

public class FileUtils {
    static final Logger logger = LogUtils.getLogger(FileUtils.class);
    static final Pattern regexAbsolutePath = Pattern.compile("^([a-zA-Z]:)?[\\\\/].*$");
    public static final File temporaryDirectory = Program.getOs() != Program.OS.MACOSX ? new File(System.getProperty("java.io.tmpdir")) : new File("/tmp");

    public static boolean canWrite(File file) {
        return file.isFile() && file.canWrite() || !file.exists() && file.getParentFile().canWrite();
    }

    public static File checkDirectory(File path, String name) throws IOException {
        return FileUtils.checkDirectory(path, name, false);
    }

    public static File checkDirectory(File path, String name, boolean create) throws IOException {
        File dir;
        name = name != null ? (String)name + " " : "";
        try {
            dir = path.getCanonicalFile();
        }
        catch (IOException e) {
            throw new IOException((String)name + path + " cannot be canonised", e);
        }
        if (dir.isDirectory()) {
            if (dir.canWrite()) {
                return dir;
            }
            throw new IOException("cannot write to " + (String)name + path);
        }
        if (dir.exists()) {
            throw new IOException((String)name + path + " is not a directory");
        }
        if (create) {
            if (!dir.mkdirs()) {
                throw new IOException("could not create " + (String)name + path);
            }
        } else {
            dir = null;
        }
        return dir;
    }

    public static boolean createDirectory(File dir) throws IOException {
        if (dir.exists()) {
            return false;
        }
        if (!dir.mkdirs()) {
            throw new IOException("could not create " + dir);
        }
        return true;
    }

    public static void copy(File source, File target) throws IOException {
        FileUtils.copy(source, target, false);
    }

    public static void copy(File source, File target, boolean append) throws IOException {
        if (!source.exists()) {
            throw new IOException(source + " does not exist");
        }
        if (!source.isFile()) {
            throw new IOException(source + " not a file");
        }
        if (!source.canRead()) {
            throw new IOException("cannot read " + source);
        }
        if (target.exists()) {
            if (!target.isFile()) {
                throw new IOException(target + " not a file");
            }
            if (!target.canWrite()) {
                throw new IOException("cannot write " + source);
            }
        } else {
            target.createNewFile();
        }
        try (FileChannel in = new FileInputStream(source).getChannel();
             FileChannel out = new FileOutputStream(target, append).getChannel();){
            out.transferFrom(in, 0L, in.size());
        }
    }

    public static void copyResource(String name, File target) throws IOException {
        int length;
        InputStream in = Thread.currentThread().getContextClassLoader().getResourceAsStream(name);
        if (in == null) {
            throw new IOException(name + " not a system resource");
        }
        if (target.exists()) {
            if (!target.isFile()) {
                throw new IOException(target + " not a file");
            }
            if (!target.canWrite()) {
                throw new IOException("cannot write " + target);
            }
        }
        FileOutputStream out = new FileOutputStream(target);
        byte[] buffer = new byte[1024];
        while ((length = in.read(buffer)) >= 0) {
            ((OutputStream)out).write(buffer, 0, length);
        }
        in.close();
        ((OutputStream)out).close();
    }

    public static String getSuffix(File file) {
        String name = file.getName();
        int i = name.lastIndexOf(46);
        return i++ > 0 ? name.substring(i) : null;
    }

    public static String getSuffix(File file, String defaultSuffix) {
        String suffix = FileUtils.getSuffix(file);
        return StringUtils.isNullOrEmpty(suffix) ? defaultSuffix : suffix;
    }

    public static String read(File file) throws IOException {
        return new String(Files.readAllBytes(file.toPath()), StandardCharsets.UTF_8);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String readFirstLine(File file) {
        BufferedReader in;
        String line = null;
        try {
            in = new BufferedReader(new FileReader(file));
        }
        catch (FileNotFoundException e) {
            logger.error("could not open {}", (Object)file);
            return null;
        }
        try {
            line = in.readLine();
        }
        catch (IOException e) {
            logger.error("could not read {}", (Object)file);
        }
        finally {
            try {
                in.close();
            }
            catch (IOException e) {
                logger.error("could not close {}", (Object)file);
            }
        }
        return line;
    }

    public static void write(File file, String ... lines) throws IOException {
        PrintStream out = new PrintStream(file);
        for (String line : lines) {
            out.println(StringUtils.trimEnd(line));
        }
        out.close();
    }

    public static boolean deleteDirectory(File directory) {
        if (!directory.exists()) {
            logger.error("directory {} does not exist", (Object)directory);
            return false;
        }
        if (!directory.isDirectory()) {
            logger.error("{} is not a directory", (Object)directory);
            return false;
        }
        if (directory.getName().equals(".") || directory.getName().equals("..")) {
            logger.error("cannot delete relative directory {}", (Object)directory);
            return false;
        }
        File[] files = directory.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isFile()) {
                    if (file.delete()) continue;
                    logger.error("could not delete {}", (Object)file);
                    continue;
                }
                if (file.getName().equals(".") || file.getName().equals("..")) continue;
                FileUtils.deleteDirectory(file);
            }
        }
        if (!directory.delete()) {
            logger.error("could not delete {}", (Object)directory);
        }
        return true;
    }

    public static File createFile(String name, String ... dirs) throws IOException {
        File file = FileUtils.findFile(name, true, dirs);
        if (!file.exists()) {
            file = file.createNewFile() ? file : null;
        }
        return file;
    }

    public static File findFile(String name, String ... dirs) {
        return FileUtils.findFile(name, false, dirs);
    }

    public static File findFile(String name, boolean readWrite, String ... dirs) {
        File file = new File(FileUtils.expandHome(name));
        if (file.isAbsolute()) {
            if (file.exists()) {
                return file.canRead() && (!readWrite || file.canWrite()) ? file : null;
            }
            return readWrite && file.getParentFile().canWrite() ? file : null;
        }
        File newFile = null;
        for (String dir : dirs) {
            if (StringUtils.isNullOrEmpty(dir)) continue;
            file = dir.endsWith(File.separator) || dir.endsWith(File.separator + ".") ? new File(dir + name) : new File(dir + File.separator + name);
            if ((file = file.getAbsoluteFile()).exists()) {
                if (!file.canRead() || readWrite && !file.canWrite()) continue;
                return file;
            }
            if (!readWrite || newFile != null || !file.getParentFile().canWrite()) continue;
            newFile = file;
        }
        return newFile;
    }

    public static String expandHome(String path) {
        if (path.startsWith("~/")) {
            return System.getProperty("user.home", "~") + path.substring(1);
        }
        return path;
    }

    public static boolean isAbsolute(File file) {
        return FileUtils.isAbsolute(file.getPath());
    }

    public static boolean isAbsolute(String path) {
        return regexAbsolutePath.matcher(path).matches();
    }

    public static boolean isOwner(File file) {
        boolean owner = false;
        if (file.exists()) {
            try {
                UserPrincipal fileOwner = Files.getOwner(file.toPath(), new LinkOption[0]);
                Path tmpFile = Files.createTempFile(null, null, new FileAttribute[0]);
                UserPrincipal tmpOwner = Files.getOwner(tmpFile, new LinkOption[0]);
                Files.delete(tmpFile);
                owner = fileOwner.equals(tmpOwner);
            }
            catch (IOException e) {
                Localise.logError(logger, "Cannot determine owner of %1$s - %2$s", file.toString(), e.getMessage());
            }
        }
        return owner;
    }

    public static File stripRoot(File path, File root) {
        return FileUtils.stripRoot(path, root, false);
    }

    public static File stripRoot(File path, File root, boolean relative) {
        String pathString = path.toString();
        String rootString = root.toString();
        int rootStringLength = rootString.length() + (relative ? 1 : 0);
        if (pathString.startsWith(rootString)) {
            return pathString.length() > rootStringLength ? new File(pathString.substring(rootStringLength)) : null;
        }
        return path;
    }
}

